%% 
% Estimated theory spectrum on all the sites of the triangle (for arbitrary 
% transmission rates scat.t1 and scat.t2)
%%

clear all;
close all;

%%

% Gamma of each peak
% gamma_no_scat = [200 175 175 175 175 180 180 195 200 200 ...
%     220 240]*1e3;

%% Some additional parameters related to experiment

% Constants
par.const.hbar = 6.62607004e-34/(2*pi);
par.const.c = 299792458;
par.const.T = 300;
par.const.kB = 1.38064852e-23;

% Parameters for mechanics
par.mech.omega = 2*pi*330e6;
par.mech.amech = 1.570496020158940e-05;
par.mech.length = 3*28;
par.mech.uc_meff = 2e-15;
par.mech.strip_meff = 2.0655e-14;
par.mech.meff = par.mech.strip_meff*par.mech.length;
par.mech.nb = par.const.kB*par.const.T/(par.const.hbar*par.mech.omega);

% Parameters for optics
par.opt.det = par.mech.omega;
par.opt.laser_power = 120e-6; % in Watts
par.opt.lambda = 1564.0562e-9;
par.opt.Qi = 163e3;
par.opt.Qe = 860e3;
par.opt.Qt = 137e3;

par.opt.omega = 2*pi*par.const.c/par.opt.lambda;
par.opt.a_in = sqrt(par.opt.laser_power/(par.const.hbar*par.opt.omega));
par.opt.Ki = par.opt.omega/par.opt.Qi;
par.opt.Ke = par.opt.omega/par.opt.Qe;
% par.opt.K = par.opt.Ki + par.opt.Ke;
par.opt.K = par.opt.omega/par.opt.Qt;
par.opt.nc = par.opt.Ke/2*par.opt.a_in^2/(par.opt.det^2 + ...
    par.opt.K^2/4);
par.opt.alpha = sqrt(par.opt.Ke/2)*par.opt.a_in/...
    (1i*par.opt.det - par.opt.K/2);

% OM and geometry parameters
% par.OM.unit_cell_g0 = 2*pi/sqrt(2)*sqrt(33^2 + 8^2)*1e3;
% par.OM.g0 = par.OM.unit_cell_g0/sqrt(par.mech.meff/par.mech.uc_meff);

%% Scattering matrix analysis

% Load band structure data values
load('./Data/experiment_compare_data_horizontal.mat');
load('./Data/experiment_compare_data_slant.mat');

% Get the range of energy values for green region
energy_min = max(min(data_horz.energy), min(data_slant.energy));
energy_max = min(max(data_horz.energy), max(data_slant.energy));

% Get the range of energy values for gray region
energy_gray_min = min(data_slant.energy);
energy_gray_max = energy_min;

% Shift in spectrum
shift_freq = -3.523725502690544e6;

% Range of values of energy.green to check out for peak position
num_energy = 5e6;
energy.green = linspace(energy_min, energy_max, num_energy);
energy.gray = linspace(energy_gray_min, energy_gray_max, num_energy);

% Spectrum energy.green values
ene_spe_min = 321e6; % Range obtained from experiment data
ene_spe_max = 336.9e6;
ene_spe = linspace(ene_spe_min - shift_freq, ene_spe_max - shift_freq, ...
    1e3);

% Find the k and g0 value for each energy via interpolation
kinterp.green.horz = interp1(data_horz.energy, data_horz.k, ...
    energy.green);
kinterp.green.slant = interp1(data_slant.energy, data_slant.k, ...
    energy.green);

kinterp.gray.slant = interp1(data_slant.energy, data_slant.k, ...
    energy.gray);

g0interp.green.horz = interp1(data_horz.energy, data_horz.g0, ...
    energy.green);
g0interp.green.slant = interp1(data_slant.energy, ...
    data_slant.g0, energy.green);

g0interp.gray.slant = interp1(data_slant.energy, ...
    data_slant.g0, energy.gray, 'spline');

% Get g(beta) for each energy
tri.amech = 1.570496020158940e-05;
tri.L = 25.480436788; % side length
tri.realL = 28;
tri.amech_corr = pi/data_horz.k(end);
tri.Leff_corr = tri.L*tri.amech/tri.amech_corr;

% Reflection and transmission co-efficients
scat.t1 = sqrt(0.9);
scat.r1 = 1i*sqrt(1 - abs(scat.t1)^2);
scat.t2 = scat.t1;
scat.r2 = 1i*sqrt(1 - abs(scat.t2)^2);

% Get peak positions in green region
[peak_pos.green, kpeak.green, vel.green, g0peak.green] = ...
    get_peak_pos(kinterp.green, g0interp.green, tri, ...
    scat, energy.green);

% Get peak positions in gray region
[peak_pos.gray, kpeak.gray, vel.gray, g0peak.gray] = ...
    get_peak_pos_gray(energy_gray_min, energy_gray_max, ...
    tri, data_slant, scat);

% Mechanical damping factor Gamma (assume same for all the peaks)
Gamma.green = 200e3*ones(1, size(peak_pos.green,2));
Gamma.gray = 200e3*ones(1, size(peak_pos.gray,2));

% gamma_no_scat = [200 200 200 200 200 200 200 200 200 200 ...
%     200 200]*1e3;
% for i=1:size(gamma_no_scat,2)
%     Gamma(2*i-1) = gamma_no_scat(i);
%     Gamma(2*i) = gamma_no_scat(i);
% end

%% Calculating spectrum

num_points = ceil(tri.Leff_corr);
start = rem(tri.Leff_corr,1)/2;
finish = tri.Leff_corr - rem(tri.Leff_corr,1)/2;
dis_temp = linspace(start, finish, num_points)*tri.amech_corr;

dis_array = [dis_temp dis_temp+tri.Leff_corr*tri.amech_corr ...
    dis_temp+2*tri.Leff_corr*tri.amech_corr];

% Every site on the triangle
dis = dis_array;

% Every 2nd site on the triangle
% dis = dis_array(1:2:end);

y2 = zeros(size(ene_spe, 2), size(dis,2));

temp_val = (par.opt.Ke*par.opt.K/4*...
    abs(par.opt.alpha*chi_opt(ene_spe*2*pi,par) - ...
    conj(par.opt.alpha*chi_opt(-ene_spe*2*pi,par))).^2);

for i=1:size(dis,2)
    % Get g0 at the measurement node
    g0_mech_cav.green = get_g0_mech_cav(kpeak.green, scat, dis(i), ...
        tri, vel.green, g0peak.green);
    g0_mech_cav.green = g0_mech_cav.green/sqrt(par.mech.length);

    g0_mech_cav.gray = get_g0_mech_cav_gray(kpeak.gray, scat, ...
        dis(i), tri, vel.gray, g0peak.gray);
    g0_mech_cav.gray = g0_mech_cav.gray/sqrt(2/3*par.mech.length);

    % Ignoring the optical backaction
    y2(:,i) = calc_SII_analytical(ene_spe, peak_pos, Gamma, ...
        g0_mech_cav, par);
    y2(:,i) = y2(:,i)./temp_val(1,:)';
end

%% Plotting spectrum

% Minimum and maximum value of y-axis
% basevalue = min(abs(y2));
basevalue = 0;
maxvalue = 5e-3;
colour_area = [92/256, 206/256, 205/256];

figure()
hold on;
pbaspect([1.2389 1 1])
set(gca,'FontSize',18)

xlim([ene_spe_min ene_spe_max])
ylim([0.5 75.5])
% xlabel('Frequency');
% ylabel('C S_{xx}/S_{xx}^{(SQL)}');
box on;

x = [ene_spe_min, ene_spe_max];
y = [0.5, 75.5];
image(x, y, transpose(y2),'CDataMapping','scaled')
colorbar;

%% Function definitions

    function y = chi_opt(w,par)
        y = 1./(par.opt.K/2 - 1i.*(w + par.opt.det));
    end